// SPDX-License-Identifier: GPL-2.0
/*
 * xHCI host controller driver
 *
 * Copyright (C) 2008 Intel Corp.
 *
 * Author: Sarah Sharp
 * Some code borrowed from the Linux EHCI driver.
 */

#include "xhci.h"

char *xhci_get_slot_state(struct xhci_hcd *xhci,
		struct xhci_container_ctx *ctx)
{
	struct xhci_slot_ctx *slot_ctx = xhci_get_slot_ctx(xhci, ctx);
	int state = GET_SLOT_STATE(le32_to_cpu(slot_ctx->dev_state));

	return xhci_slot_state_string(state);
}

void xhci_dbg_trace(struct xhci_hcd *xhci, void (*trace)(struct va_format *),
			const char *fmt, ...)
{
	struct va_format vaf;
	va_list args;

	va_start(args, fmt);
	vaf.fmt = fmt;
	vaf.va = &args;
	xhci_dbg(xhci, "%pV\n", &vaf);
	trace(&vaf);
	va_end(args);
}
EXPORT_SYMBOL_GPL(xhci_dbg_trace);

/* Debugfs interface */
#include <linux/debugfs.h>
static struct dentry *xhci_debug_root;

#if 0
int xhci_debugfs_init(void)
{
	xhci_debug_root = debugfs_create_dir("xhci", usb_debug_root);
	if (!xhci_debug_root)
		return -ENOENT;

	return 0;
}
#endif

void xhci_debugfs_cleanup(void)
{
	if (xhci_debug_root)
		debugfs_remove(xhci_debug_root);
}

//#ifdef CONFIG_XHCI_TEST_REGISTERS
#include <linux/uaccess.h>
#include <linux/ctype.h>

static unsigned test_reg_offset = 0;

/* output: OFFSET VALUE */
static ssize_t debug_test_regs_read(struct file *file,
	       char __user *user_buf, size_t count, loff_t *ppos)
{
	struct usb_bus	*bus = file->private_data;
	struct xhci_hcd	*xhci = hcd_to_xhci(bus_to_hcd(bus));
	unsigned long	flags;
	char		buf[32];
	size_t		size;

	spin_lock_irqsave (&xhci->lock, flags);
	size = snprintf(buf, 31, "0x%x 0x%x\n",
			test_reg_offset,
			readl((void *)xhci->cap_regs + test_reg_offset));
	spin_unlock_irqrestore (&xhci->lock, flags);

	return simple_read_from_buffer(user_buf, count, ppos, buf, size);
}

/* input: OFFSET [VALUE] */
static ssize_t debug_test_regs_write(struct file *file,
	       const char __user *user_buf, size_t count, loff_t *ppos)
{
	struct usb_bus	*bus = file->private_data;
	struct xhci_hcd	*xhci = hcd_to_xhci(bus_to_hcd(bus));
	unsigned long	flags;
	char		buf[64], *p;
	unsigned long	offs, val;
	char		**args;
	int		nargs;
	int		ret = -EINVAL;

	/* Check length and copy the user buffer */
	if (count > sizeof(buf) - 1)
		return -E2BIG;
	if (copy_from_user(buf, user_buf, count))
		return -EFAULT;

	/* Null termination process */
	buf[63] = '\0';
	p = strchr(buf, '\n');
	if (p)
		*p = '\0';

	args = argv_split(GFP_KERNEL, buf, &nargs);
	if (!args)
		return -ENOMEM;
	if (nargs > 2)
		goto out;

	/* Get the offset of register */
	if (kstrtoul(args[0], 0, &offs) < 0)
		goto out;
	if (offs & 0x3)	/* The offset must be 4byte-aligned */
		goto out;
	if (nargs == 1)
		goto end;

	/* Get the setting value of register */
	if (kstrtoul(args[1], 0, &val) < 0)
		goto out;

	/* Set the value to the register */
	spin_lock_irqsave (&xhci->lock, flags);
	/* Use xhci->cap_regs because it is directly mapped to hcd->regs */
	writel((u32)val, (void *)xhci->cap_regs + offs);
	spin_unlock_irqrestore (&xhci->lock, flags);

end:
	test_reg_offset = offs;
	ret = count;
out:
	argv_free(args);

	return count;
}

static int debug_test_regs_open(struct inode *inode, struct file *file)
{
	file->private_data = inode->i_private;

	return 0;
}

static const struct file_operations debug_test_regs_fops = {
	.owner		= THIS_MODULE,
	.open		= debug_test_regs_open,
	.read		= debug_test_regs_read,
	.write		= debug_test_regs_write,
	.llseek		= default_llseek,
};
//#endif

void xhci_create_debug_files(struct xhci_hcd *xhci)
{
	struct usb_bus *bus = &xhci_to_hcd(xhci)->self;
	struct dentry *debug_dir;

	debug_dir = debugfs_create_dir(bus->bus_name, xhci_debug_root);
	if (!debug_dir)
		return;
	xhci->debug_dir = debug_dir;

	if (!debugfs_create_file("test_regs", 0600, xhci->debug_dir, bus,
						    &debug_test_regs_fops))
		goto file_error;
	/* TODO: Add interfaces for other registers? (op_regs, ir_set, etc) */
	return;

file_error:
	debugfs_remove_recursive(xhci->debug_dir);
}

void xhci_remove_debug_files(struct xhci_hcd *xhci)
{
	debugfs_remove_recursive(xhci->debug_dir);
	xhci->debug_dir = NULL;
}
